import 'package:flutter/material.dart';
import 'package:flutter_riverpod/flutter_riverpod.dart';
import 'package:flutter_screenutil/flutter_screenutil.dart';
import 'package:gap/gap.dart';
import 'package:razinsoft_rider/core/extensions/extensions.dart';
import 'package:razinsoft_rider/core/theme/color_palette.dart';
import 'package:razinsoft_rider/core/utils/app_colors.dart';
import 'package:razinsoft_rider/generated/l10n.dart';
import 'package:razinsoft_rider/presentation/wallet/widgets/wallet_balance.dart';
import '../../../common/error_view.dart';
import '../../../common/loading_view.dart';
import '../../../core/routes/app_routes.dart';
import '../../../data/models/my_card_model/my_card_model.dart';
import '../../../data/services/navigation_service.dart';
import '../../account_page/provider/theme_provider.dart';
import '../provider/provider.dart';

class WalletPage extends ConsumerStatefulWidget {
  const WalletPage({super.key});

  @override
  ConsumerState<WalletPage> createState() => _WalletPageState();
}

class _WalletPageState extends ConsumerState<WalletPage> {
  final TextEditingController customAmountController = TextEditingController();

  @override
  void initState() {
    super.initState();
    Future.microtask(() {
      ref.read(walletsNotifierProvider.notifier).getWallets();
      ref.read(myCardsProvider.notifier).myCards();
    });
  }

  @override
  void dispose() {
    super.dispose();
    customAmountController.dispose();
  }

  bool isDark() => ref.read(themeModeProvider.notifier).isDarkMode();
  @override
  Widget build(BuildContext context) {
    final walletState = ref.watch(walletsNotifierProvider);

    return Scaffold(
      appBar: AppBar(
        title: Text(
          AppLocalizations.of(context).wallet,
          style: context.bodyMedium?.copyWith(
            fontSize: 16.sp,
            fontWeight: FontWeight.w500,
            color: isDark() ? Colors.white : const Color(0xFF24262D),
          ),
        ),
        centerTitle: true,
      ),
      resizeToAvoidBottomInset: true,
      body: SafeArea(
        child: RefreshIndicator(
          onRefresh: () =>
              ref.read(walletsNotifierProvider.notifier).getWallets(),
          child: LayoutBuilder(
            builder: (context, constraints) => SingleChildScrollView(
              physics: const AlwaysScrollableScrollPhysics(),
              padding: EdgeInsets.only(
                bottom: MediaQuery.of(context).viewInsets.bottom,
              ),
              child: ConstrainedBox(
                constraints: BoxConstraints(minHeight: constraints.maxHeight),
                child: Column(
                  crossAxisAlignment: CrossAxisAlignment.start,
                  children: [
                    Gap(8.h),
                    walletState.when(
                      initial: () => const SizedBox(),
                      loading: () => const LoadingView(),
                      error: (e) => ErrorView(message: e.message),
                      success: (walletModel) {
                        final data = walletModel.data;
                        if (data == null) {
                          return Center(
                            child: Text(
                              AppLocalizations.of(
                                context,
                              ).no_wallet_data_available,
                              style: context.bodyMedium?.copyWith(
                                color: Colors.red,
                              ),
                            ),
                          );
                        }

                        return Container(
                          color: isDark() ? AppColors.surface : Colors.white,
                          padding: EdgeInsets.all(16.r),
                          child: walletBalance(
                            context,
                            amount: walletModel.data?.balance,
                            customAmountController: customAmountController,
                          ),
                        );
                      },
                    ),
                    Gap(8.h),
                    Container(
                      color: isDark() ? AppColors.surface : Colors.white,
                      padding: EdgeInsets.all(16.r),
                      child: Row(
                        children: [
                          Expanded(
                            child: Text(
                              AppLocalizations.of(context).payment_gateway,
                              style: context.bodyMedium?.copyWith(
                                fontSize: 18.sp,
                                fontWeight: FontWeight.w600,
                                color: isDark()
                                    ? Colors.white
                                    : const Color(0xFF24262D),
                              ),
                            ),
                          ),
                          Gap(16.w),
                          InkWell(
                            onTap: () {
                              NavigationService.pushNamed(
                                AppRoutes.addPaymentGateway,
                              );
                            },
                            child: Container(
                              padding: EdgeInsets.symmetric(
                                horizontal: 12.w,
                                vertical: 9.h,
                              ),
                              decoration: BoxDecoration(
                                borderRadius: BorderRadius.circular(8.r),
                                border: Border.all(
                                  color: ColorPalette.primary50,
                                ),
                              ),
                              child: Row(
                                children: [
                                  Text(
                                    AppLocalizations.of(context).add_new,
                                    style: context.bodyMedium?.copyWith(
                                      fontSize: 14.sp,
                                      fontWeight: FontWeight.w500,
                                      color: ColorPalette.primary50,
                                    ),
                                  ),
                                  Gap(4.w),
                                  Icon(
                                    Icons.add,
                                    color: ColorPalette.primary50,
                                    size: 15.h,
                                  ),
                                ],
                              ),
                            ),
                          ),
                        ],
                      ),
                    ),
                    Consumer(
                      builder: (context, ref, _) {
                        final state = ref.watch(myCardsProvider);
                        ref.watch(deleteCardProvider);

                        return state.when(
                          initial: () => const SizedBox.shrink(),
                          loading: () => const LoadingView(),
                          success: (list) {
                            if (list.isEmpty) {
                              return Padding(
                                padding: EdgeInsets.all(16.r),
                                child: Center(
                                  child: Text(
                                    AppLocalizations.of(context).no_cards_yet,
                                    style: context.bodyLarge?.copyWith(
                                      color: Colors.red,
                                    ),
                                  ),
                                ),
                              );
                            }
                            return Container(
                              color: isDark() ? Colors.black12 : Colors.white,
                              padding: EdgeInsets.symmetric(
                                horizontal: 16.w,
                                vertical: 16.h,
                              ),
                              child: ListView.builder(
                                shrinkWrap: true,
                                padding: EdgeInsets.zero,
                                physics: const NeverScrollableScrollPhysics(),
                                itemCount: list.length,
                                itemBuilder: (context, index) {
                                  final Cards card = list[index];
                                  final deleteState = ref.watch(
                                    deleteCardProvider,
                                  );
                                  final deleteNotifier = ref.read(
                                    deleteCardProvider.notifier,
                                  );

                                  final isDeleting = deleteState.maybeWhen(
                                    loading: () =>
                                        deleteNotifier.deletingCardId ==
                                        card.id,
                                    orElse: () => false,
                                  );
                                  final bool isLast = index == list.length - 1;
                                  return Container(
                                    margin: EdgeInsets.only(
                                      bottom: isLast ? 0 : 16.h,
                                    ),
                                    decoration: BoxDecoration(
                                      borderRadius: BorderRadius.circular(8.r),
                                      color: isDark()
                                          ? Colors.black12
                                          : const Color(0xFFF1F7FE),
                                      border: isDark()
                                          ? Border.all(color: Colors.white)
                                          : null,
                                    ),
                                    padding: EdgeInsets.all(16.r),
                                    child: Row(
                                      children: [
                                        Expanded(
                                          child: Text(
                                            card.name?.capitalize() ?? 'N/A',
                                            style: context.bodyMedium?.copyWith(
                                              fontSize: 24.sp,
                                              fontWeight: FontWeight.bold,
                                              color: ColorPalette.primary50,
                                            ),
                                          ),
                                        ),
                                        Gap(16.w),
                                        InkWell(
                                          onTap: isDeleting
                                              ? null
                                              : () {
                                                  if (card.id == null) return;
                                                  ref
                                                      .read(
                                                        deleteCardProvider
                                                            .notifier,
                                                      )
                                                      .deleteCard(id: card.id);
                                                },
                                          child: isDeleting
                                              ? SizedBox(
                                                  height: 15.h,
                                                  width: 15.w,
                                                  child: const Center(
                                                    child:
                                                        CircularProgressIndicator(),
                                                  ),
                                                )
                                              : Icon(
                                                  Icons.delete_forever,
                                                  color: Colors.red,
                                                  size: 24.h,
                                                ),
                                        ),
                                      ],
                                    ),
                                  );
                                },
                              ),
                            );
                          },
                          error: (e) => ErrorView(message: e.message),
                        );
                      },
                    ),
                  ],
                ),
              ),
            ),
          ),
        ),
      ),
    );
  }
}
